
----------------------------------------
-- DateSet-Cache & Komponenten-Listen --
----------------------------------------



/***********************************************************************************************************************
*****   BASE : CACHE   *************************************************************************************************
***********************************************************************************************************************/

-- Berechnet eine Änderungs-Prüfsumme für die Tabelle, um entscheiden zu können, ob eine gecachte Tabelle noch aktuell ist. AddWhere gibt eine optionale Einschränkung der Cache an.
CREATE OR REPLACE FUNCTION GetCacheHash(TableName VARCHAR, AddWhere VARCHAR DEFAULT NULL) RETURNS INTEGER AS $$
DECLARE id VARCHAR;
BEGIN
  -- Bei Änderungen ebenfalls TCimDM_SysDB.GetCacheHash in UDataModul_SysDB anpassen!
  BEGIN
    IF TRIM(AddWhere) = '' THEN
      AddWhere := NULL;
    END IF;
    EXECUTE 'SELECT COUNT(dbrid) || '' '' || COALESCE(MAX(dbrid), '''') || '' '' || COALESCE(MAX(GREATEST(modified_date, insert_date::TIMESTAMP(0)))::VARCHAR, '''') '
      || 'FROM ' || TableName || COALESCE(' WHERE ' || AddWhere, '') INTO id;
  EXCEPTION WHEN OTHERS THEN
    id := currenttime()::VARCHAR;
  END;
  RETURN ('x' || substr(md5(id), 1, 8))::BIT(32)::INTEGER;
END $$ LANGUAGE plpgsql;



CREATE OR REPLACE FUNCTION GetCacheHashFromSelect(CachedSelect VARCHAR) RETURNS INTEGER AS $$
DECLARE id VARCHAR;
BEGIN
  -- Bei Änderungen ebenfalls TCimDM_SysDB.GetCacheHash in UDataModul_SysDB anpassen!
  BEGIN
    EXECUTE 'SELECT COUNT(dbrid) || '' '' || COALESCE(MAX(dbrid), '''') || '' '' || COALESCE(MAX(GREATEST(modified_date, insert_date::TIMESTAMP(0)))::VARCHAR, '''') '
      || 'FROM (' || CachedSelect || ') AS cached' INTO id;
  EXCEPTION WHEN OTHERS THEN
    id := currenttime()::VARCHAR;
  END;
  RETURN ('x' || substr(md5(id), 1, 8))::BIT(32)::INTEGER;
END $$ LANGUAGE plpgsql;



/***Cache***
-- sendet allen PRODATs eine Update-Message
CREATE OR REPLACE FUNCTION SendCacheUpdate(TableName VARCHAR, CacheHash VARCHAR DEFAULT NULL) RETURNS VOID AS $$
DECLARE Message VARCHAR;
BEGIN
  Message := '"Table=' || TableName || '";"CacheHash=' || REPLACE(COALESCE(CacheHash, GetCacheHash(TableName)), '"', '""') || '"';
  PERFORM PRODAT_COMM_FUNKTION('CacheUpdate', Message);
END $$ LANGUAGE plpgsql;
*/



/***********************************************************************************************************************
*****   BASE : ITEM TYPES   ********************************************************************************************
***********************************************************************************************************************/

-- mögliche Typen für Tabelle "itemlist"
CREATE TABLE itemtypes
( it_type           VARCHAR(10) PRIMARY KEY,  -- Typ der Daten (z.B. "CompOption", "GridFilter", "HelpComp" oder "HelpIndex")
  it_table          VARCHAR(50) NOT NULL,     -- Name der Tabellen, worin die Daten liegen
  it_idfield        VARCHAR(20) NOT NULL,     -- Name des ID-Feldes [FIELDS: co_id, gf_id, hi_id]
  it_itemfield      VARCHAR(20) NOT NULL,     -- Name des Feldes, welches auf il_id verweist [FIELDS: co_item, gf_item, hi_item]
  it_deletedfield   VARCHAR(20),              -- Name des Feldes, welches anzeigt ob dieser Eintrag gelöscht wurde [FIELDS: gf_deleted, hi_deleted]
  it_groupname      VARCHAR(30) NOT NULL,     -- Caption/Name für il_group
  it_itemname       VARCHAR(30),              -- Caption/Name für il_item (wenn NULL, dann wird il_item nicht verwendet)
  it_description    TEXT                      -- Beschreibung
);

CREATE UNIQUE INDEX it_id_index ON itemtypes (LOWER(it_type));  -- case-insensitive UniqueCheck für it_type

-- Einträge in itemtypes, für nachfolgende Tabellen, werden in "Z Predefined.sql" erstellt.



/***********************************************************************************************************************
*****   BASE : ITEM LIST   *********************************************************************************************
***********************************************************************************************************************/

-- Liste aller verwendeten Items/Komponenten
CREATE TABLE itemlist
( il_id             SERIAL PRIMARY KEY,                          -- -
  il_type           VARCHAR(10)  NOT NULL REFERENCES itemtypes,  -- -
  il_group          VARCHAR(100) NOT NULL,                       -- Fensterklasse/Gruppe
  il_item           VARCHAR(250)                                 -- Komponentenname/Teil (z.B. ein CimEdit oder GridView)

  -- Auskommentiert, da SubSelect nicht möglich
  --, CONSTRAINT check_item CHECK (il_item IS NULL OR (SELECT it_itemname IS NOT NULL FROM itemtypes WHERE it_type = il_type))  -- Fehler wenn il_item gesetzt, aber nicht verlangt (it_itemname=NULL))
);

-- case-insensitiver UniqueCheck für il_type+il_group+il_item (HelpIndex und HelpComp werden gemeinsam als Eines geprüft)
CREATE UNIQUE INDEX il_data_index ON itemlist ((CASE WHEN il_type='HelpIndex' THEN 'HelpComp' ELSE il_type END), LOWER(il_group), COALESCE(LOWER(il_item), ''));



/***Cache***
-- sendet allen PRODATs eine Update-Message für die entsprechende Tabelle
CREATE OR REPLACE FUNCTION itemlist_cacheupdate() RETURNS TRIGGER AS $$
BEGIN
  PERFORM SendCacheUpdate((SELECT it_table FROM itemtypes WHERE it_type = new.il_type));
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER itemlist_cacheupdate
AFTER UPDATE
ON itemlist
FOR EACH ROW
WHEN (old.il_group <> new.il_group OR old.il_item <> new.il_item)
EXECUTE PROCEDURE itemlist_cacheupdate();
*/



-- Liest die ID eines Items aus und erstellt Dieses, falls benötigt und noch nicht vorhanden.
CREATE OR REPLACE FUNCTION itemlist_getitem(Type VARCHAR, GroupName VARCHAR, ItemName VARCHAR, CreateIfNotExists BOOLEAN DEFAULT true) RETURNS INTEGER AS $$
DECLARE ID INTEGER;
BEGIN
  IF ItemName = '' THEN
    ItemName := NULL;
  END IF;
  SELECT il_id INTO ID FROM itemlist WHERE il_type = Type AND LOWER(il_group) = LOWER(GroupName) AND LOWER(COALESCE(il_item, '')) = LOWER(COALESCE(ItemName, ''));
  IF ID IS NULL AND CreateIfNotExists THEN
    INSERT INTO itemlist (il_type, il_group, il_item) VALUES (Type, GroupName, ItemName) RETURNING il_id INTO ID;
  END IF;
  RETURN ID;
END $$ LANGUAGE plpgsql;



-- berechnet eine ID für die Synchronisierung (auch über die referenzierten Tabellen) => Basisfunktion für abhängige Tabellen
CREATE OR REPLACE FUNCTION itemdata_syncid(id INTEGER, add VARCHAR DEFAULT NULL) RETURNS VARCHAR AS $$
  SELECT LOWER(il_type) || ':' || LOWER(il_group)
    || IfThen(il_item IS NULL, ':*', ':' || LOWER(il_item))
    || IfThen($2 IS NULL, '', ':' || $2)
  FROM itemlist
  WHERE il_id = $1
$$ LANGUAGE SQL;



-- prüft, ob es für eine Datensatz in itemlist mit einem/mehreren Datensätze in der verlinkten Tabelle gibt
-- true = ja | false = nein | null = nur Gelöschtes vorhanden
CREATE OR REPLACE FUNCTION itemdata_islinked(id INTEGER) RETURNS BOOLEAN AS $$
DECLARE
  _table        VARCHAR;
  _itemfield    VARCHAR;
  _deletedfield VARCHAR;
  all_found     BOOLEAN;
  real_found    BOOLEAN;
BEGIN
  -- Gibt ob es ein Verlinktes Feld für die Suche?
  SELECT it_table, it_itemfield, it_deletedfield
  INTO _table, _itemfield, _deletedfield
  FROM itemlist
  JOIN itemtypes ON it_type = il_type
  WHERE il_id = id;

  EXECUTE
    IfThen(_deletedfield IS NOT NULL,
      ' SELECT count(*) > 0 AS all_found, count(IfThen(' || quote_ident(_deletedfield) || ', NULL, true)) > 0 AS real_found ',
      ' SELECT count(*) > 0 AS all_found, count(*) > 0 AS real_found ') ||
    ' FROM ' || quote_ident(_table) ||
    ' WHERE ' || quote_ident(_itemfield) || ' = $1'
  USING id INTO all_found, real_found;

  RETURN IfThen(real_found, true, IfThen(all_found, NULL, false));
END$$ LANGUAGE plpgsql;



/***********************************************************************************************************************
*****   COMP OPTION   **************************************************************************************************
***********************************************************************************************************************/

-- [SYNCRO-TABLE] globale und benutzerbezogene Einstellungen für Komponenten
CREATE TABLE component_options
( co_id      SERIAL PRIMARY KEY,                                         -- [it_idfield]      [SYNCRO:NotThisFields]
  co_item    INTEGER NOT NULL REFERENCES itemlist ON DELETE CASCADE,     -- [it_itemfield]    [SYNCRO:SyncID=il_type+il_group+il_item+co_name] Komponente (der GridView)
  co_minr    INTEGER REFERENCES llv ON DELETE CASCADE ON UPDATE CASCADE, -- [data *]          für benutzerabhängige Optionen
  co_name    VARCHAR(100),                                               -- [data]            [SYNCRO:SyncID=il_type+il_group+il_item+co_name] Option-Name
  co_value   TEXT                                                        -- [data]            Wert der Option (String, Integer, usw.)
--co_sync    BOOLEAN NOT NULL DEFAULT false,                             -- [data *]          in die Synchronisierung aufnehmen
--co_deleted BOOLEAN NOT NULL DEFAULT false,                             -- [it_deletedfield] [SYNCRO:Deleted]
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0),                                          --                   [SYNCRO:Modified]
  --
--CONSTRAINT check_sync CHECK (co_minr IS NULL OR NOT co_sync)           -- persönliche Filter können/dürfen nicht synchronisiert werden
-- ACHTUNG: nicht vergessen co_deleted in Tabelle "itemtypes" zu registrieren -> "Z Predefined.sql" und DBUpdates
);

-- case-insensitiver UniqueCheck für co_item+co_minr+co_name
CREATE UNIQUE INDEX co_data_index ON component_options (co_item, COALESCE(co_minr, -2), COALESCE(LOWER(co_name), ''));
CREATE INDEX component_options_co_minr ON component_options (co_minr);

--- #21443 [Log] Änderungslog für Dashboardänderungen (Component Options)
CREATE OR REPLACE FUNCTION component_options__b_ud() RETURNS TRIGGER AS $$
 DECLARE _operation    varchar(1);
 BEGIN

    IF tg_op = 'DELETE' THEN
        _operation := 'D';
    ELSE
        _operation := 'U';
    END IF;

    INSERT INTO component_options_log(
     col_operation, col_co_id, col_co_item, col_co_minr, col_co_name, col_co_value, insert_by    , insert_date    , modified_by    , modified_date
  )
  VALUES (
      _operation  , old.co_id, old.co_item, old.co_minr, old.co_name, old.co_value, old.insert_by, old.insert_date, old.modified_by, old.modified_date
         );

  IF tg_op = 'DELETE' THEN
        RETURN old;
    ELSE                      -- UPDATE
        RETURN new;
    END IF;

 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER component_options__b_ud
    BEFORE UPDATE OR DELETE
    ON component_options
    FOR EACH ROW
    EXECUTE PROCEDURE component_options__b_ud();


/***Cache***
-- sendet allen PRODATs eine Update-Message für die Tabelle component_options
CREATE OR REPLACE FUNCTION component_options_cacheupdate() RETURNS TRIGGER AS $$
BEGIN
  PERFORM SendCacheUpdate('component_options');
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER component_options_cacheupdate
AFTER INSERT OR UPDATE OR DELETE
ON component_options
FOR EACH STATEMENT
EXECUTE PROCEDURE component_options_cacheupdate();
*/

--- #23570
CREATE OR REPLACE FUNCTION TSystem.util_dfm_unescape_basic(_txt text)
RETURNS text
LANGUAGE plpgsql
AS $$
DECLARE
  out text;
BEGIN
  -- Zeilenumbrüche aus Delphi: "#13#10" -> LF
  out := replace(coalesce(_txt, ''), '#13#10', E'\n');

  -- Feste Zuordnung gemäß Vorgabe:
  -- #214 -> Ü, #246 -> ü, #220 -> Ö, #252 -> ö, #196 -> Ä, #228 -> ä
  out := replace(out, '''#214''', 'Ö');
  out := replace(out, '''#246''', 'ö');
  out := replace(out, '''#220''', 'Ü');
  out := replace(out, '''#252''', 'ü');
  out := replace(out, '''#196''', 'Ä');
  out := replace(out, '''#228''', 'ä');
  out := replace(out, '''#223''', 'ß');

  RETURN out;
END;
$$;
---
CREATE OR REPLACE FUNCTION TSystem.Search_Dashboard(
    _p_search_strings   text,
    _p_case_insensitive boolean
)
RETURNS TABLE (
    elementID      varchar,
    elementName    varchar,
    description    varchar,
    matching_terms varchar,
    co_value       text,
    co_name        varchar,
    il_item        varchar
) AS $$
    -- Zweck:
    --   Suche in component_options nach Treffern in co_value und co_name,
    --   gefiltert auf Datensaetze, die das Dashboard betreffen.
    --
    -- Hinweise:
    --   * Finale Anweisung MUSS ein einzelnes SELECT sein (SQL-Funktion).
    --   * Case-Sensitivity wird per _p_case_insensitive gesteuert.
    --   * matching_terms aggregiert eindeutige Regex-Treffer aus co_value.

    WITH base AS (
      SELECT
          co_id::varchar         AS elementID,
          co_name::varchar       AS elementName,
          ('object ' || co_name) AS description,
          --co_value::text         AS co_value,
          tsystem.util_dfm_unescape_basic(co_value)::text         AS co_value,   --- Umlaute und Zeilenumbruche 'übersetzen'
          il_item::varchar       AS il_item
      FROM component_options
        LEFT JOIN itemlist     ON il_id = co_item
      WHERE co_value ILIKE '%TCimDashboardFrame%'
    ),
    filtered AS (
      SELECT *
      FROM base b
      WHERE
        CASE
          WHEN _p_case_insensitive THEN coalesce(b.co_value,'')    ~* _p_search_strings
          ELSE                        coalesce(b.co_value,'')     ~  _p_search_strings
        END
        OR
        CASE
          WHEN _p_case_insensitive THEN coalesce(b.elementName,'') ~* _p_search_strings
          ELSE                        coalesce(b.elementName,'')  ~  _p_search_strings
        END
    )
    SELECT
        f.elementID,
        coalesce(f.elementName, 'leer?!?') AS elementName,
        f.description,
        -- Eindeutige Treffer aus co_value zusammenfassen (kann NULL sein, wenn keine Matches)
        (
          SELECT string_agg(DISTINCT m, ',')
          FROM (
            SELECT array_to_string(
                     regexp_matches(
                       f.co_value,
                       _p_search_strings,
                       CASE WHEN _p_case_insensitive THEN 'gi' ELSE 'g' END
                     ),
                     ','
                   ) AS m
          ) s
        ) AS matching_terms,
        f.co_value AS co_value,
        f.elementName AS co_name,
        f.il_item AS il_item
    FROM filtered f
    ORDER BY f.elementID;

$$ LANGUAGE sql SECURITY DEFINER;
---

CREATE OR REPLACE FUNCTION SetCompOption(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, Value VARCHAR, MitNr INTEGER DEFAULT current_user_minr()) RETURNS VOID AS $$
BEGIN
  IF length(GroupName) > 100 THEN  -- itemlist.il_group
    RAISE NOTICE 'GetCompOption(%, %, %): GroupName zu lang (%)',  GroupName, ItemName, Option, length(GroupName);
  END IF;
  IF length(ItemName) > 100 THEN  -- itemlist.il_item
    RAISE NOTICE 'GetCompOption(%, %, %): ItemName zu lang (%)',   GroupName, ItemName, Option, length(ItemName);
  END IF;
  IF length(Option) > 100 THEN  -- component_options.co_name
    RAISE NOTICE 'GetCompOption(%, %, %): OptionName zu lang (%)', GroupName, ItemName, Option, length(Option);
  END IF;
--IF length(Value) > 200 THEN  -- component_options.co_value
--  RAISE NOTICE 'GetCompOption(%, %, %): Value zu lang (%)',      GroupName, ItemName, Option, length(Value);
--END IF;
  IF ItemName = '' THEN  -- itemlist.il_item
    ItemName := NULL;
  END IF;
  IF Option = '' THEN  -- component_options.co_name
    Option := NULL;
  END IF;

  UPDATE component_options SET co_value = Value
  WHERE co_item = itemlist_getitem('CompOption', GroupName, ItemName)
    AND co_minr IS NOT DISTINCT FROM IfThen(MitNr < 0, NULL, MitNr) AND LOWER(co_name) IS NOT DISTINCT FROM LOWER(Option);

  IF NOT FOUND THEN
    INSERT INTO component_options (co_item, co_minr, co_name, co_value)
    VALUES (itemlist_getitem('CompOption', GroupName, ItemName), IfThen(MitNr < 0, NULL, MitNr), Option, Value);
  END IF;
END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION SetCompOption(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, Value BOOLEAN, MitNr INTEGER DEFAULT current_user_minr()) RETURNS VOID AS $$
BEGIN
  PERFORM SetCompOption(GroupName, ItemName, Option, Value::INT::VARCHAR, MitNr);
END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION SetCompOption(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, Value INTEGER, MitNr INTEGER DEFAULT current_user_minr()) RETURNS VOID AS $$
BEGIN
  PERFORM SetCompOption(GroupName, ItemName, Option, Value::VARCHAR, MitNr);
END $$ LANGUAGE plpgsql;



CREATE OR REPLACE FUNCTION GetCompOption(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, MitNr INTEGER DEFAULT current_user_minr(), DefaultValue VARCHAR DEFAULT '') RETURNS VARCHAR AS $$
DECLARE
  Result VARCHAR;
BEGIN
  IF length(GroupName) > 100 THEN  -- itemlist.il_group
    RAISE NOTICE 'GetCompOption(%, %, %): GroupName zu lang (%)',  GroupName, ItemName, Option, length(GroupName);
  END IF;
  IF length(ItemName) > 100 THEN  -- itemlist.il_item
    RAISE NOTICE 'GetCompOption(%, %, %): ItemName zu lang (%)',   GroupName, ItemName, Option, length(ItemName);
  END IF;
  IF length(Option) > 100 THEN  -- component_options.co_name
    RAISE NOTICE 'GetCompOption(%, %, %): OptionName zu lang (%)', GroupName, ItemName, Option, length(Option);
  END IF;
  IF ItemName = '' THEN  -- itemlist.il_item
    ItemName := NULL;
  END IF;
  IF Option = '' THEN  -- component_options.co_name
    Option := NULL;
  END IF;

  SELECT co_value INTO Result FROM component_options
  WHERE co_item = itemlist_getitem('CompOption', GroupName, ItemName, false)
    AND co_minr IS NOT DISTINCT FROM IfThen(MitNr < 0, NULL, MitNr) AND LOWER(co_name) IS NOT DISTINCT FROM LOWER(Option);

  RETURN COALESCE(IfThen(FOUND, Result, DefaultValue), '');
END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION GetCompOptionB(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, MitNr INTEGER DEFAULT current_user_minr(), DefaultValue BOOLEAN DEFAULT false) RETURNS BOOLEAN AS $$
DECLARE
  Result VARCHAR;
BEGIN
  Result := GetCompOption(GroupName, ItemName, Option, MitNr, COALESCE(DefaultValue, false)::INT::VARCHAR);
  BEGIN
    RETURN Result::INTEGER <> 0;
  EXCEPTION
    WHEN OTHERS THEN
      RETURN IfThen(LOWER(Result) IN ('true', 'false'), Result ILIKE 'true', COALESCE(DefaultValue, false));
  END;
END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION GetCompOptionI(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, MitNr INTEGER DEFAULT current_user_minr(), DefaultValue INTEGER DEFAULT 0) RETURNS INTEGER AS $$
DECLARE
  Result VARCHAR;
BEGIN
  Result := GetCompOption(GroupName, ItemName, Option, MitNr, COALESCE(DefaultValue, 0)::VARCHAR);
  BEGIN
    RETURN Result::INTEGER;
  EXCEPTION
    WHEN OTHERS THEN
      RETURN COALESCE(DefaultValue, 0);
  END;
END $$ LANGUAGE plpgsql;

-- TODO : Überlegen MitNr mit NULL als Default vorzubelegen, so wie es in UDataModul_SysDB auch gemacht ist.
-- > aktuell werden die SQL-Funktion noch nirgends verwendet

-- TODO : siehe UDataModul_SysDB.ExistsCompOption
-- CREATE OR REPLACE FUNCTION ExistsCompOption(GroupName VARCHAR, ItemName VARCHAR, Option VARCHAR, MitNr INTEGER DEFAULT current_user_minr(), CheckValue INTEGER DEFAULT '*') RETURNS BOOLEAN AS $$



/***********************************************************************************************************************
*****   GRID FILTER   **************************************************************************************************
***********************************************************************************************************************/

-- [SYNCRO-TABLE] Filter für CimGridView
CREATE TABLE gridfilter
( gf_id             TIMESTAMP(0) PRIMARY KEY DEFAULT currenttime(),              -- [it_idfield]      [SYNCRO:NotThisFields]
  gf_item           INTEGER NOT NULL REFERENCES itemlist ON DELETE CASCADE,      -- [it_itemfield]    [SYNCRO:SyncID=il_type+il_group+il_item+gf_description] Komponente (der GridView)
  gf_minr           INTEGER REFERENCES llv ON DELETE CASCADE ON UPDATE CASCADE,  -- [data *]          für persönliche Benutzerfilter
  gf_description    VARCHAR(50),                                                 -- [data *]          [SYNCRO:SyncID=il_type+il_group+il_item+gf_description] dieser Name wird z.B. im GridFilter-Popup angezeigt
  gf_filter         BYTEA,                                                       -- [data]            binärer FilterText
  gf_filterhint     VARCHAR(200),                                                -- [data]            Anfang des menschenlesbaren FilterTextes (dieser kann leider nicht ins Grid geladen werden)
  gf_default        BOOLEAN NOT NULL DEFAULT false,                              -- [data]            der Standard-Filter
  gf_sync           BOOLEAN NOT NULL DEFAULT false,                              -- [data *]          in die Synchronisierung aufnehmen
  gf_deleted        BOOLEAN NOT NULL DEFAULT false,                              -- [it_deletedfield] [SYNCRO:Deleted]
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0),                                                  --                   [SYNCRO:Modified]
  --
  CONSTRAINT check_sync    CHECK (gf_minr IS NULL OR NOT gf_sync),               -- persönliche Filter können/dürfen nicht synchronisiert werden
  CONSTRAINT check_default CHECK (gf_minr IS NULL OR NOT gf_default)             -- persönliche Filter können/dürfen keine Standardfilter sein
);

-- case-insensitiver UniqueCheck für gf_item+gf_minr+gf_description
CREATE UNIQUE INDEX gf_data_index ON gridfilter (gf_item, COALESCE(gf_minr, -2), TSystem.IFTHEN(gf_deleted, NULL, COALESCE(LOWER(gf_description), '')));



-- Setzt Default der anderen Filter zurück, wenn neues Default gesetzt wird.
CREATE OR REPLACE FUNCTION gridfilter_defaultupdate() RETURNS TRIGGER AS $$
BEGIN
  UPDATE gridfilter SET gf_default = false
  WHERE gf_item = new.gf_item AND gf_id <> new.gf_id AND new.gf_default AND gf_default;
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER gridfilter_defaultupdate
BEFORE INSERT OR UPDATE
ON gridfilter
FOR EACH ROW
WHEN (new.gf_default)
EXECUTE PROCEDURE gridfilter_defaultupdate();



/***Cache***
-- sendet allen PRODATs eine Update-Message für die Tabelle gridfilter
CREATE OR REPLACE FUNCTION gridfilter_cacheupdate() RETURNS TRIGGER AS $$
BEGIN
  PERFORM SendCacheUpdate('gridfilter');
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER gridfilter_cacheupdate
AFTER INSERT OR UPDATE OR DELETE
ON gridfilter
FOR EACH STATEMENT
EXECUTE PROCEDURE gridfilter_cacheupdate();
*/



/***********************************************************************************************************************
*****   GRID LAYOUT   **************************************************************************************************
***********************************************************************************************************************/

-- [SYNCRO-TABLE] Layout für CimGridView
CREATE TABLE gridlayout
( gl_id             SERIAL PRIMARY KEY,                                          -- [it_idfield]      [SYNCRO:NotThisFields]
  gl_item           INTEGER NOT NULL REFERENCES itemlist ON DELETE CASCADE,      -- [it_itemfield]    [SYNCRO:SyncID=il_type+il_group+il_item+gl_detailview] Komponente (der GridView)
  gl_minr           INTEGER REFERENCES llv ON DELETE CASCADE ON UPDATE CASCADE,  -- [data *]          für persönliche Benutzerfilter (NULL = globales Layout, wird synchronisiert)
  gl_detailview     VARCHAR(50),                                                 -- [data]            [SYNCRO:SyncID=il_type+il_group+il_item+gl_detailview] NULL=HauptView, Rest=DetailView
  gl_layout         BYTEA,                                                       -- [data]            Layout (neu=binär, alt_importe=INI)
--gl_sync           BOOLEAN NOT NULL DEFAULT false,                              -- [data *]          in die Synchronisierung aufnehmen
  gl_deleted        BOOLEAN NOT NULL DEFAULT false,                              -- [it_deletedfield] [SYNCRO:Deleted]
  gl_default        BOOLEAN NOT NULL DEFAULT true
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0),                                                  --                   [SYNCRO:Modified]
  --
--CONSTRAINT check_sync CHECK (gl_minr IS NULL OR NOT gl_sync)                   -- persönliche Filter können/dürfen nicht synchronisiert werden
);

-- case-insensitiver UniqueCheck für gl_item+gl_minr+gl_description
CREATE UNIQUE INDEX gl_data_index ON gridlayout (gl_item, COALESCE(gl_minr, -2), TSystem.IFTHEN(gl_deleted, NULL, COALESCE(LOWER(gl_detailview), '')));
CREATE INDEX gridlayout_gl_minr ON gridlayout (gl_minr);


--
CREATE OR REPLACE FUNCTION gridlayout__a_iu__default() RETURNS TRIGGER AS $$
BEGIN
  PERFORM disablemodified();
  --- #20158 Layouts bei allen Benutzern zu globalem setzen. Automatisch beim Insert oder beim Update wenn gl_default von Admin auf TRUE gesetzt wurde
  UPDATE
    gridlayout
  SET
    gl_default = FALSE
  WHERE
    gl_minr IS NOT NULL
    AND gl_item = new.gl_item
    AND gl_default;
  PERFORM enablemodified();
  RETURN NULL;
END $$ LANGUAGE plpgsql;
--
CREATE TRIGGER gridlayout__a_iu__default
  AFTER INSERT OR UPDATE
  ON gridlayout
  FOR EACH ROW
  WHEN (new.gl_default AND new.gl_minr IS NULL) --nur für globale Layouts
  EXECUTE PROCEDURE gridlayout__a_iu__default();
--


/***********************************************************************************************************************
*****   HELP INDEX   ***************************************************************************************************
***********************************************************************************************************************/

-- [SYNCRO-TABLE] Hilfe-Index
CREATE TABLE helpindex
( hi_id           SERIAL PRIMARY KEY,                                     -- [it_idfield]      [SYNCRO:NotThisFields]
  hi_item         INTEGER NOT NULL REFERENCES itemlist ON DELETE CASCADE, -- [it_itemfield]    [SYNCRO:SyncID=il_type+il_group+il_item] Komponente
  hi_helplink     VARCHAR(150),                                           -- [data]            Indexname in der Hilfe
  hi_anker        VARCHAR(20),                                            -- [data]            Sprungziel (Scrollpunkt #) in der Hilfedatei
  hi_deleted      BOOLEAN NOT NULL DEFAULT false                          -- [it_deletedfield] [SYNCRO:Deleted]
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0)                                            --                   [SYNCRO:Modified]
);

CREATE UNIQUE INDEX hi_data_index ON helpindex (hi_item);



/***Cache***
-- sendet allen PRODATs eine Update-Message für die Tabelle helpindex
CREATE OR REPLACE FUNCTION helpindex_cacheupdate() RETURNS TRIGGER AS $$
BEGIN
  PERFORM SendCacheUpdate('helpindex');
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER helpindex_cacheupdate
AFTER INSERT OR UPDATE OR DELETE
ON helpindex
FOR EACH STATEMENT
EXECUTE PROCEDURE helpindex_cacheupdate();
*/



/***********************************************************************************************************************
*****   ShortCuts   ****************************************************************************************************
***********************************************************************************************************************/

-- [SYNCRO-TABLE] dynamische ShortCuts
CREATE TABLE Keyboard_ShortCuts
( kbsc_id        SERIAL PRIMARY KEY,          -- [SYNCRO:NotThisFields]
  kbsc_KeyCode   INTEGER NOT NULL,            -- [SYNCRO:SyncID=kbsc_KeyCode+kbsc_KeyState+kbsc_Kunde] VK_-Code
  kbsc_KeyState  INTEGER NOT NULL DEFAULT 4,  -- [SYNCRO:SyncID=kbsc_KeyCode+kbsc_KeyState+kbsc_Kunde] Shift=1 Alt=2 Ctrl=4 AltGr=2+4
  kbsc_KeyValue  VARCHAR(20) NOT NULL,
  kbsc_Comment   TEXT,
  kbsc_Kunde     VARCHAR(20),                 -- [SYNCRO:SyncID=kbsc_KeyCode+kbsc_KeyState+kbsc_Kunde] [SYNCRO:Kunde]
  kbsc_Deleted   BOOL NOT NULL DEFAULT False  -- [SYNCRO:Deleted]
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0)                -- [SYNCRO:Modified]
);

CREATE UNIQUE INDEX Keyboard_ShortCuts_Index ON Keyboard_ShortCuts (kbsc_KeyCode, kbsc_KeyState, COALESCE(LOWER(kbsc_Kunde), ''));


CREATE OR REPLACE RULE Keyboard_ShortCuts_DeleteRule AS ON DELETE
TO Keyboard_ShortCuts WHERE  NOT TSystem.current_user_in_syncro_dblink()  AND NOT old.kbsc_Deleted
DO INSTEAD UPDATE Keyboard_ShortCuts SET kbsc_Deleted = TRUE WHERE kbsc_id = old.kbsc_id;


CREATE OR REPLACE FUNCTION Keyboard_ShortCuts__b_iu() RETURNS TRIGGER AS $$
BEGIN
  IF TG_OP = 'INSERT' AND TG_WHEN = 'BEFORE' THEN
    -- gelöschte Datensätze sind nicht im DataSet enthalten - um den Duplicate-Error zu unterbinden, Diesen jetzt löschen, falls vorhanden
    DELETE FROM Keyboard_ShortCuts
    WHERE kbsc_Deleted AND kbsc_KeyCode = new.kbsc_KeyCode AND kbsc_KeyState = new.kbsc_KeyState AND kbsc_Kunde IS NOT DISTINCT FROM new.kbsc_Kunde;
  END IF;

  IF TG_OP = 'UPDATE' AND TG_WHEN = 'BEFORE' THEN
    IF old.kbsc_KeyCode <> new.kbsc_KeyCode OR old.kbsc_KeyState <> new.kbsc_KeyState OR old.kbsc_Kunde IS DISTINCT FROM new.kbsc_Kunde THEN
      IF TG_WHEN = 'BEFORE' THEN
        -- alten bereits gelöschten Datensatz entfernen
        DELETE FROM Keyboard_ShortCuts
        WHERE kbsc_Deleted AND kbsc_KeyCode = new.kbsc_KeyCode AND kbsc_KeyState = new.kbsc_KeyState AND kbsc_Kunde IS NOT DISTINCT FROM new.kbsc_Kunde;
      END IF;
      IF TG_WHEN = 'AFTER' THEN  -- erst im AFTER, da sonst der Originaldatensatz noch exisitert (Kopie+Original=Keyboard_ShortCuts_Index)
        -- Kopie des Datensatzes als gelöscht speichern (vor der Änderung) -> für Syncronisation mit altem Name+Kunde
        INSERT INTO Keyboard_ShortCuts (kbsc_KeyCode, kbsc_KeyState, kbsc_KeyValue, kbsc_Comment, kbsc_Kunde, kbsc_Deleted, insert_by, insert_date, modified_by, modified_date)
        VALUES (old.kbsc_KeyCode, old.kbsc_KeyState, old.kbsc_KeyValue, old.kbsc_Comment, old.kbsc_Kunde, True, old.insert_by, old.insert_date, old.modified_by, old.modified_date);
      END IF;
    END IF;
  END IF;

  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER Keyboard_ShortCuts__b_iu
BEFORE INSERT OR UPDATE  -- DELETE siehe RULE Keyboard_ShortCuts_DeleteRule
ON Keyboard_ShortCuts
FOR EACH ROW
EXECUTE PROCEDURE Keyboard_ShortCuts__b_iu();

CREATE TRIGGER Keyboard_ShortCuts__a_u
AFTER UPDATE
ON Keyboard_ShortCuts
FOR EACH ROW
EXECUTE PROCEDURE Keyboard_ShortCuts__b_iu();



/***********************************************************************************************************************
*****   Keyboard-Layout   **********************************************************************************************
***********************************************************************************************************************/
-- http://redmine.prodat-sql.de/issues/4179

-- [SYNCRO-TABLE] Tastatur-Layout
CREATE TABLE Keyboard_Layouts
( kblay_id            SERIAL PRIMARY KEY,          -- [SYNCRO:NotThisFields]
  kblay_Name          VARCHAR(100) NOT NULL,       -- [SYNCRO:SyncID=kblay_Name+kblay_Kunde]
  kblay_Data          TEXT,
  kblay_Kunde         VARCHAR(20),                 -- [SYNCRO:SyncID=kblay_Name+kblay_Kunde] [SYNCRO:Kunde]
  kblay_Deleted       BOOL NOT NULL DEFAULT False  -- [SYNCRO:Deleted]
  -- System (tables__generate_missing_fields)
  --modified_date TIMESTAMP(0)                     -- [SYNCRO:Modified]
);

CREATE UNIQUE INDEX Keyboard_Layouts_Index ON Keyboard_Layouts (LOWER(kblay_Name), COALESCE(LOWER(kblay_Kunde), ''));


CREATE OR REPLACE RULE Keyboard_Layouts_DeleteRule AS ON DELETE
TO Keyboard_Layouts WHERE  NOT TSystem.current_user_in_syncro_dblink()  AND NOT old.kblay_Deleted
DO INSTEAD UPDATE Keyboard_Layouts SET kblay_Deleted = TRUE WHERE kblay_id = old.kblay_id;


CREATE OR REPLACE FUNCTION Keyboard_Layouts__b_iu() RETURNS TRIGGER AS $$
BEGIN
  IF TG_OP = 'INSERT' AND TG_WHEN = 'BEFORE' THEN
    -- gelöschte Datensätze sind nicht im DataSet enthalten - um den Duplicate-Error zu unterbinden, Diesen jetzt löschen, falls vorhanden
    DELETE FROM Keyboard_Layouts
    WHERE kblay_Deleted AND kblay_Name = new.kblay_Name AND kblay_Kunde IS NOT DISTINCT FROM new.kblay_Kunde;
  END IF;

  IF TG_OP = 'UPDATE' THEN
    IF old.kblay_Name IS DISTINCT FROM new.kblay_Name OR old.kblay_Kunde IS DISTINCT FROM new.kblay_Kunde THEN
      IF TG_WHEN = 'BEFORE' THEN
        -- alten bereits gelöschten Datensatz entfernen
        DELETE FROM Keyboard_Layouts
        WHERE kblay_Deleted AND kblay_Name = new.kblay_Name AND kblay_Kunde IS NOT DISTINCT FROM new.kblay_Kunde;
      END IF;
      IF TG_WHEN = 'AFTER' THEN  -- erst im AFTER, da sonst der Originaldatensatz noch exisitert (Kopie+Original=Keyboard_Layouts_Index)
        -- Kopie des Datensatzes als gelöscht speichern (vor der Änderung) -> für Syncronisation mit altem Name+Kunde
        INSERT INTO Keyboard_Layouts (kblay_Name, kblay_Data, kblay_Kunde, kblay_Deleted, insert_by, insert_date, modified_by, modified_date)
        VALUES (old.kblay_Name, old.kblay_Data, old.kblay_Kunde, True, old.insert_by, old.insert_date, old.modified_by, old.modified_date);
      END IF;
    END IF;
  END IF;

  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER Keyboard_Layouts__b_iu
BEFORE INSERT OR UPDATE  -- DELETE siehe RULE Keyboard_Layouts_DeleteRule
ON Keyboard_Layouts
FOR EACH ROW
EXECUTE PROCEDURE Keyboard_Layouts__b_iu();

CREATE TRIGGER Keyboard_Layouts__a_u
AFTER UPDATE
ON Keyboard_Layouts
FOR EACH ROW
EXECUTE PROCEDURE Keyboard_Layouts__b_iu();


-- Softwarelinks für Steuerung von anderer Software
-- siehe: https://redmine.prodat-sql.de/projects/prodat-v12-public/wiki/Softwarelink_-_Softlink
CREATE TABLE softlink (
  slink_id         serial NOT NULL PRIMARY KEY,
  slink_bez_textno integer,                -- Textnr. Übersetzung für Beschreibung
  slink_bez        varchar(100),           -- Beschreibung
  -- todo / klärung: slink_table_schema: siehe PR#191. Anmerkungen zu Client, public usw: https://github.com/prodat/psql/pull/191#discussion_r414365718
  slink_table      VARCHAR(50) NOT NULL,   -- Quell-Tabelle z.B.: 'auftg', 'art', 'akd'

  slink_type       varchar(4) NOT NULL,    -- Ausführungsart, z.B.: 'EXEC' oder 'LINK'

  -- Ausführung, z.B.: 'S:\elo_test.bat artikel=<#ak_nr#>' oder 'https://redmine.prodat-sql.de/issues/<#prj_nr#>'
  slink_app        varchar(255) NOT NULL,

  -- Bedingung z.B.: "(:ag_astat IN ('E', 'R'))" oder "(:parentabk IS NOT NULL)"
  slink_cond       text,
  slink_sql        text,                   -- Verlinkung auf assoziierte Tabellen
  slink_order      integer                 -- Sortierreinfolge

  CONSTRAINT softlink__slink_bez__slink_textno__not_null CHECK (
         slink_bez IS NOT NULL
      OR slink_bez_textno IS NOT NULL
  ),

  CONSTRAINT softlink__slink_type__validate CHECK(
      slink_type IN ( 'EXEC', 'LINK' )
  ),

  CONSTRAINT softlink__table__existing CHECK (
      -- checks for existing table within the current searchpath
      to_regclass( slink_table ) IS NOT NULL
  )
);

--- #21443 [Log] Änderungslog für Dashboardänderungen (Component Options)
CREATE TABLE component_options_log(
  col_operation        varchar(1),                        --- 'U' oder 'D'  entsprechend TG_OP des auslösenden Triggers
  col_co_id            integer,
  col_co_item          integer,
  col_co_minr          integer,
  col_co_name          varchar(100),
  col_co_value         text
);
---